<?php
/**
 * Payment Gateway Currency for WooCommerce - Core Class
 *
 * @version 1.5.0
 * @since   1.0.0
 *
 * @author  Algoritmika Ltd.
 */

if ( ! defined( 'ABSPATH' ) ) exit; // Exit if accessed directly

if ( ! class_exists( 'Alg_WC_Payment_Gateways_by_Currency_Core' ) ) :

class Alg_WC_Payment_Gateways_by_Currency_Core {

	/**
	 * Constructor.
	 *
	 * @version 1.5.0
	 * @since   1.0.0
	 *
	 * @todo    [next] Convert price & currency: fix chosen payment method: uncomment || remove || make optional?
	 * @todo    [next] Convert price & currency: currency reports
	 * @todo    [maybe] Advanced: add option to add all hooks on `init` action
	 */
	function __construct() {
		// Restrict (i.e. allowed/denied) currencies
		if ( 'yes' === get_option( 'alg_wc_payment_gateways_by_currency_plugin_enabled', 'yes' ) ) {
			add_filter( 'woocommerce_available_payment_gateways',       array( $this, 'restrict_payment_gateways' ), PHP_INT_MAX );
		}
		// Convert price & currency
		if ( 'yes' === get_option( 'alg_wc_pgbc_convert_currency_enabled', 'no' ) ) {
			$this->convert_rates = get_option( 'alg_wc_pgbc_convert_rate',              array() );
			$extra_options       = get_option( 'alg_wc_pgbc_convert_currency_advanced', array() );
			// Product price
			add_filter( 'woocommerce_product_get_price',                array( $this, 'convert_price' ), PHP_INT_MAX, 2 );
			add_filter( 'woocommerce_product_variation_get_price',      array( $this, 'convert_price' ), PHP_INT_MAX, 2 );
			// Shipping price
			if ( ! isset( $extra_options['shipping'] ) || 'yes' === $extra_options['shipping'] ) {
				add_filter( 'woocommerce_package_rates',                array( $this, 'convert_shipping_price' ), PHP_INT_MAX, 2 );
			}
			// Coupons
			if ( ! isset( $extra_options['coupon'] ) || 'yes' === $extra_options['coupon'] ) {
				add_filter( 'woocommerce_coupon_get_amount',            array( $this, 'convert_coupon_amount' ), PHP_INT_MAX, 2 );
			}
			if ( ! isset( $extra_options['coupon_min_amount'] ) || 'yes' === $extra_options['coupon_min_amount'] ) {
				add_filter( 'woocommerce_coupon_get_minimum_amount',    array( $this, 'convert_price' ), PHP_INT_MAX );
			}
			if ( ! isset( $extra_options['coupon_max_amount'] ) || 'yes' === $extra_options['coupon_max_amount'] ) {
				add_filter( 'woocommerce_coupon_get_maximum_amount',    array( $this, 'convert_price' ), PHP_INT_MAX );
			}
			// Cart fees
			if ( isset( $extra_options['cart_fee'] ) && 'yes' === $extra_options['cart_fee'] ) {
				add_action( 'init',                                     array( $this, 'add_convert_cart_fees_hook' ), PHP_INT_MAX );
			}
			// Currency code & symbol
			add_filter( 'woocommerce_currency',                         array( $this, 'convert_currency' ),        PHP_INT_MAX );
			add_filter( 'woocommerce_currency_symbol',                  array( $this, 'convert_currency_symbol' ), PHP_INT_MAX );
			// PayPal supported currencies
			add_filter( 'woocommerce_paypal_supported_currencies',      array( $this, 'extend_paypal_supported_currencies' ), PHP_INT_MAX );
			// Fix chosen payment method
//			add_action( 'woocommerce_checkout_update_order_review',     array( $this, 'fix_chosen_payment_method' ) );
			// Script
			add_action( 'wp_footer',                                    array( $this, 'add_checkout_script' ) );
			// Currency exchange rates
			add_action( 'alg_wc_pgbc_settings_saved',                   array( $this, 'get_currency_exchange_rates_manual' ) );
			if ( '' != get_option( 'alg_wc_pgbc_convert_currency_auto_rates_cron', '' ) ) {
				add_action( 'init',                                     array( $this, 'schedule_cron' ) );
				add_action( 'alg_wc_pgbc_currency_exchange_rates',      array( $this, 'get_currency_exchange_rates' ) );
			} else {
				add_action( 'init',                                     array( $this, 'unschedule_cron' ) );
			}
		}
	}

	/*
	 * schedule_cron.
	 *
	 * @version 1.5.0
	 * @since   1.5.0
	 *
	 * @see     https://developer.wordpress.org/reference/functions/wp_schedule_event/
	 *
	 * @todo    [next] (dev) `admin_notices`: `check_if_wp_crons_disabled`
	 * @todo    [next] (dev) debug
	 * @todo    [next] (feature) `cron_schedules`

	 */
	function schedule_cron() {
		$event_timestamp = wp_next_scheduled( 'alg_wc_pgbc_currency_exchange_rates', array( 'hourly' ) );
		if ( ! $event_timestamp ) {
			wp_schedule_event( time(), 'hourly', 'alg_wc_pgbc_currency_exchange_rates', array( 'hourly' ) );
		}
	}

	/**
	 * unschedule_cron.
	 *
	 * @version 1.5.0
	 * @since   1.5.0
	 *
	 * @see     https://developer.wordpress.org/reference/functions/wp_unschedule_event/
	 *
	 * @todo    [next] (dev) run this even if `! alg_wc_pgbc_convert_currency_enabled`
	 * @todo    [maybe] (dev) also run on plugin deactivation?
	 * @todo    [maybe] (dev) run on settings save?
	 */
	function unschedule_cron() {
		$event_timestamp = wp_next_scheduled( 'alg_wc_pgbc_currency_exchange_rates', array( 'hourly' ) );
		if ( $event_timestamp ) {
			wp_unschedule_event( $event_timestamp, 'alg_wc_pgbc_currency_exchange_rates', array( 'hourly' ) );
		}
	}

	/*
	 * get_currency_exchange_rates_manual.
	 *
	 * @version 1.5.0
	 * @since   1.5.0
	 */
	function get_currency_exchange_rates_manual() {
		if ( 'yes' === get_option( 'alg_wc_pgbc_convert_currency_auto_rates_now', 'no' ) ) {
			update_option( 'alg_wc_pgbc_convert_currency_auto_rates_now', 'no' );
			$this->get_currency_exchange_rates();
			WC_Admin_Settings::add_message( __( 'Currency exchange rates updated.', 'payment-gateways-by-currency-for-woocommerce' ) );
		}
	}

	/*
	 * get_currency_exchange_rates.
	 *
	 * @version 1.5.0
	 * @since   1.5.0
	 */
	function get_currency_exchange_rates() {
		$currencies = get_option( 'alg_wc_pgbc_convert_currency', array() );
		$rates      = get_option( 'alg_wc_pgbc_convert_rate',     array() );
		$from       = get_woocommerce_currency();
		foreach ( $currencies as $key => $currency ) {
			if ( $rate = $this->get_currency_exchange_rate_ecb( $from, $currency ) ) {
				$rates[ $key ] = $rate;
			}
		}
		update_option( 'alg_wc_pgbc_convert_rate', $rates );
		$this->convert_rates = $rates;
	}

	/*
	 * get_currency_exchange_rate_ecb.
	 *
	 * @version 1.5.0
	 * @since   1.5.0
	 */
	function get_currency_exchange_rate_ecb( $currency_from, $currency_to ) {
		if ( isset( $this->currency_exchange_rates[ $currency_from ][ $currency_to ] ) ) {
			return $this->currency_exchange_rates[ $currency_from ][ $currency_to ];
		}
		$final_rate = false;
		if ( function_exists( 'simplexml_load_file' ) ) {
			$url = 'http://www.ecb.int/stats/eurofxref/eurofxref-daily.xml';
			$xml = ( defined( 'WP_DEBUG' ) && true === WP_DEBUG ? simplexml_load_file( $url ) : @simplexml_load_file( $url ) );
			if ( isset( $xml->Cube->Cube->Cube ) ) {
				if ( 'EUR' === $currency_from ) {
					$EUR_currency_from_rate = 1;
				}
				if ( 'EUR' === $currency_to ) {
					$EUR_currency_to_rate   = 1;
				}
				foreach ( $xml->Cube->Cube->Cube as $currency_rate ) {
					$currency_rate = $currency_rate->attributes();
					if ( ! isset( $EUR_currency_from_rate ) && $currency_from == $currency_rate->currency ) {
						$EUR_currency_from_rate = ( float ) $currency_rate->rate;
					}
					if ( ! isset( $EUR_currency_to_rate )   && $currency_to   == $currency_rate->currency ) {
						$EUR_currency_to_rate   = ( float ) $currency_rate->rate;
					}
				}
				if ( isset( $EUR_currency_from_rate, $EUR_currency_to_rate ) && 0 != $EUR_currency_from_rate ) {
					$final_rate = round( $EUR_currency_to_rate / $EUR_currency_from_rate, 6 );
					$this->currency_exchange_rates[ $currency_from ][ $currency_to ] = $final_rate;
				}
			}
		}
		return $final_rate;
	}

	/**
	 * add_checkout_script.
	 *
	 * @version 1.4.0
	 * @since   1.4.0
	 *
	 * @todo    [next] add option to disable this
	 * @todo    [next] move to `class-alg-wc-pgbc-checkout.js`
	 */
	function add_checkout_script( $core ) {
		if ( is_checkout() ) {
			?><script>
				jQuery( document ).ready( function() {
					jQuery( 'body' ).on( 'change', 'input[name="payment_method"]', function() {
						jQuery( 'body' ).trigger( 'update_checkout' );
					} );
				} );
			</script><?php
		}
	}

	/**
	 * fix_chosen_payment_method.
	 *
	 * @version 1.4.0
	 * @since   1.4.0
	 */
	function fix_chosen_payment_method( $post_data ) {
		$payment_gateway            = ( empty( $_POST['payment_method'] ) ? '' : $_POST['payment_method'] );
		$available_payment_gateways = array_keys( WC()->payment_gateways->get_available_payment_gateways() );
		if ( ! empty( $available_payment_gateways ) ) {
			 if ( ! in_array( $payment_gateway, $available_payment_gateways ) ) {
				$_POST['payment_method'] = $available_payment_gateways[0];
			 }
		} else {
			$_POST['payment_method'] = '';
		}
	}

	/**
	 * get_current_gateway.
	 *
	 * @version 1.4.0
	 * @since   1.4.0
	 *
	 * @todo    [maybe] rethink this?
	 */
	function get_current_gateway() {
		if ( '' == ( $current_gateway = WC()->session->chosen_payment_method ) ) {
			if ( '' == ( $current_gateway = ( ! empty( $_REQUEST['payment_method'] ) ? $_REQUEST['payment_method'] : '' ) ) ) {
				return ( isset( $this->last_known_current_gateway ) ? $this->last_known_current_gateway : get_option( 'woocommerce_default_gateway', '' ) );
			}
		}
		$this->last_known_current_gateway = $current_gateway;
		return $current_gateway;
	}

	/**
	 * is_cart_or_checkout.
	 *
	 * @version 1.4.0
	 * @since   1.4.0
	 */
	function is_cart_or_checkout() {
		return ( is_cart() || is_checkout() );
	}

	/**
	 * extend_paypal_supported_currencies.
	 *
	 * @version 1.4.0
	 * @since   1.4.0
	 */
	function extend_paypal_supported_currencies( $supported_currencies ) {
		if ( $this->is_cart_or_checkout() && ( $current_gateway = $this->get_current_gateway() ) ) {
			$currencies = get_option( 'alg_wc_pgbc_convert_currency', array() );
			if ( isset( $currencies[ $current_gateway ] ) && '' !== $currencies[ $current_gateway ] && ! in_array( $currencies[ $current_gateway ], $supported_currencies ) ) {
				$supported_currencies[] = $currencies[ $current_gateway ];
			}
		}
		return $supported_currencies;
	}

	/**
	 * convert_currency.
	 *
	 * @version 1.4.0
	 * @since   1.4.0
	 */
	function convert_currency( $currency ) {
		if ( $this->is_cart_or_checkout() && ( $current_gateway = $this->get_current_gateway() ) ) {
			$currencies = get_option( 'alg_wc_pgbc_convert_currency', array() );
			if ( isset( $currencies[ $current_gateway ] ) && '' !== $currencies[ $current_gateway ] ) {
				return $currencies[ $current_gateway ];
			}
		}
		return $currency;
	}

	/**
	 * convert_currency_symbol.
	 *
	 * @version 1.4.0
	 * @since   1.4.0
	 *
	 * @todo    [next] merge with `convert_currency()`
	 */
	function convert_currency_symbol( $currency_symbol ) {
		if ( $this->is_cart_or_checkout() && ( $current_gateway = $this->get_current_gateway() ) ) {
			$currency_symbols = get_option( 'alg_wc_pgbc_convert_symbol', array() );
			if ( isset( $currency_symbols[ $current_gateway ] ) && '' !== $currency_symbols[ $current_gateway ] ) {
				return $currency_symbols[ $current_gateway ];
			}
		}
		return $currency_symbol;
	}

	/**
	 * convert_shipping_package_rates.
	 *
	 * @version 1.4.0
	 * @since   1.4.0
	 */
	function convert_shipping_package_rates( $package_rates, $multiplier ) {
		$modified_package_rates = array();
		foreach ( $package_rates as $id => $package_rate ) {
			if ( 1 != $multiplier && ! empty( $package_rate->cost ) ) {
				$package_rate->cost = $package_rate->cost * $multiplier;
				if ( ! empty( $package_rate->taxes ) ) {
					$rate_taxes = $package_rate->taxes;
					foreach ( $rate_taxes as &$tax ) {
						if ( $tax ) {
							$tax *= $multiplier;
						}
					}
					$package_rate->taxes = $rate_taxes;
				}
			}
			$modified_package_rates[ $id ] = $package_rate;
		}
		return $modified_package_rates;
	}

	/**
	 * convert_shipping_price.
	 *
	 * @version 1.4.0
	 * @since   1.4.0
	 */
	function convert_shipping_price( $package_rates, $package ) {
		if ( $this->is_cart_or_checkout() && ( $current_gateway = $this->get_current_gateway() ) ) {
			if ( isset( $this->convert_rates[ $current_gateway ] ) && '' !== $this->convert_rates[ $current_gateway ] ) {
				return $this->convert_shipping_package_rates( $package_rates, $this->convert_rates[ $current_gateway ] );
			}
		}
		return $package_rates;
	}

	/**
	 * convert_coupon_amount.
	 *
	 * @version 1.4.0
	 * @since   1.4.0
	 */
	function convert_coupon_amount( $amount, $coupon ) {
		return ( $coupon->is_type( array( 'fixed_cart', 'fixed_product' ) ) ? $this->convert_price( $amount ) : $amount );
	}

	/**
	 * add_convert_cart_fees_hook.
	 *
	 * @version 1.4.0
	 * @since   1.4.0
	 *
	 * @note    `convert_cart_fees()` hook should be added as late as possible, so that all cart fees would be already added at this point
	 */
	function add_convert_cart_fees_hook() {
		add_action( 'woocommerce_cart_calculate_fees', array( $this, 'convert_cart_fees' ), PHP_INT_MAX );
	}

	/**
	 * convert_cart_fees.
	 *
	 * @version 1.4.0
	 * @since   1.4.0
	 *
	 * @todo    [maybe] find an alternative way of implementing this?
	 */
	function convert_cart_fees( $cart ) {
		if ( $current_gateway = $this->get_current_gateway() ) {
			if ( isset( $this->convert_rates[ $current_gateway ] ) && '' !== $this->convert_rates[ $current_gateway ] ) {
				$_fees = array();
				$fees  = $cart->fees_api()->get_fees();
				foreach ( $fees as $fee ) {
					$_fees[] = array(
						'name'      => $fee->name,
						'amount'    => $fee->amount * $this->convert_rates[ $current_gateway ],
						'taxable'   => $fee->taxable,
						'tax_class' => $fee->tax_class,
					);
				}
				$cart->fees_api()->set_fees( $_fees );
			}
		}
	}

	/**
	 * convert_price.
	 *
	 * @version 1.4.0
	 * @since   1.4.0
	 */
	function convert_price( $price, $product = false ) {
		if ( $price && $this->is_cart_or_checkout() && ( $current_gateway = $this->get_current_gateway() ) ) {
			if ( isset( $this->convert_rates[ $current_gateway ] ) && '' !== $this->convert_rates[ $current_gateway ] ) {
				$price *= $this->convert_rates[ $current_gateway ];
			}
		}
		return $price;
	}

	/**
	 * restrict_payment_gateways.
	 *
	 * @version 1.5.0
	 * @since   1.0.0
	 */
	function restrict_payment_gateways( $available_gateways ) {
		$current_currency = get_woocommerce_currency();
		$incl_currencies  = get_option( 'alg_wc_payment_gateways_by_currency_incl', array() );
		$excl_currencies  = get_option( 'alg_wc_payment_gateways_by_currency_excl', array() );
		foreach ( $available_gateways as $key => $gateway ) {
			if ( ! apply_filters( 'alg_wc_pgbc_pre_check', ( in_array( $key, array( 'cheque', 'bacs', 'cod', 'paypal' ) ) ) ) ) {
				continue;
			}
			if (
				( ! empty( $incl_currencies[ $key ] ) && ! in_array( $current_currency, $incl_currencies[ $key ] ) ) ||
				( ! empty( $excl_currencies[ $key ] ) &&   in_array( $current_currency, $excl_currencies[ $key ] ) )
			) {
				unset( $available_gateways[ $key ] );
			}
		}
		return $available_gateways;
	}

}

endif;

return new Alg_WC_Payment_Gateways_by_Currency_Core();
